<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Repositories\Eloquent\Admin\UserRepository;
use App\Http\ServicesLayer\FairbaseServices\FairbaseService;
use App\Models\User;
use App\Http\ServicesLayer\ForJawalyServices\ForJawalyService;
use App\Models\Cafe;
use App\Models\Notification;
use Carbon\Carbon;
use Illuminate\Support\Facades\Hash;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Tymon\JWTAuth\Facades\JWTAuth;
use Illuminate\Support\Facades\Hash as FacadesHash;
use App\Models\CafeOrder;
use App\Models\Car;
use App\Models\CarOrder;
use App\Models\GuideOrder;
use App\Models\Hotel;
use App\Models\HotelOrder;
use App\Models\Rating;
use App\Models\Restaurant;
use App\Models\RestaurantOrder;

class AuthController extends Controller
{
    public $user;
    public $userRepository;
    public $forJawalyService;
    public $fairbaseService;
    
    public $guideOrder;
    public $carOrder;
    public $hotelOrder;
    public $restaurantOrder;
    public $cafeOrder;
    public $car;
    public $hotel;
    public $restaurant;
    public $cafe;
    public $rating;
    
    public function __construct(
        User $user, UserRepository $userRepository, ForJawalyService $forJawalyService, FairbaseService $fairbaseService,
        GuideOrder $guideOrder, CarOrder $carOrder, HotelOrder $hotelOrder, RestaurantOrder $restaurantOrder, CafeOrder $cafeOrder,
        Car $car, Hotel $hotel, Restaurant $restaurant, Cafe $cafe, Rating $rating
    ){
        $this->user = $user;
        $this->userRepository = $userRepository;
        $this->forJawalyService = $forJawalyService;
        $this->fairbaseService = $fairbaseService;
        $this->guideOrder = $guideOrder;
        $this->carOrder = $carOrder;
        $this->hotelOrder = $hotelOrder;
        $this->restaurantOrder = $restaurantOrder;
        $this->cafeOrder = $cafeOrder;
        $this->car = $car;
        $this->hotel = $hotel;
        $this->restaurant = $restaurant;
        $this->cafe = $cafe;
        $this->rating = $rating;
        $this->middleware('auth:api', ['except' => ['checkToken', 'login', 'register', 'mobileCheck', 'regenerateCode', 'sendResetCodePassword', 'verifyResetCodePassword', 'resetPassword', 'changePassword']]);
    }
    
    public function checkToken()
    {
        try {
            $user = JWTAuth::parseToken()->authenticate();
            if (is_null($user)) {
                return responseJson(200, "success", false);
            }else {
                return responseJson(200, "success", true);
            }
        } catch (\Throwable $th) {
            return responseJson(200, "success", false);
        }
    }

    public function register(Request $request)
    {
        $validator = Validator::make($request->all(), [
            
            'name' => 'required|string|max:254|unique:users,name',
            'email' => 'required|string|max:254|unique:users,email',
            'mobile' => 'required|string|max:254|unique:users,mobile',
            'password' => 'required|confirmed|max:30',
            
            'file' => 'nullable|image|mimes:jpeg,png,jpg,gif',
            'user_type' => 'required|in:1,2,3,4,5,6',
            
            'country_id' => 'nullable|exists:countries,id',
            'ibna' => 'nullable|max:255',
            'logo' => 'nullable|image|mimes:jpeg,png,jpg,gif',
            'documented_files' => 'nullable|array',
            'documented_files.*' => 'required|image|mimes:jpeg,png,jpg,gif,webp',
            
            'day_price' => 'required_if:user_type,2|max:254', 
            'hour_price' => 'required_if:user_type,2|max:254', 
            'info' => 'required_if:user_type,2|max:25000', 
            'experience' => 'required_if:user_type,2|max:25000', 
            'interests' => 'required_if:user_type,2|array',
            'interests.*' => 'required_if:user_type,2|max:254',
            'tours_type' => 'required_if:user_type,2|array',
            'tours_type.*' => 'required_if:user_type,2|max:254',
            'languages' => 'required_if:user_type,2|array',
            'languages.*' => 'required_if:user_type,2|exists:languages,id',
            
            'days' => 'required|array',
            'days.*.day' => 'required|in:السبت,الاحد,الاثنين,الثلاثاء,الاربعاء,الخميس,الجمعه',
            'days.*.start' => 'nullable|date_format:H:i',
            'days.*.end' => 'nullable|date_format:H:i',
            'days.*.day_status' => 'nullable|in:0,1,2',

            'fcm_token' => 'nullable|string',
        ]);
        if ($validator->fails()) {
            return responseJson(400, "Bad Request", $validator->errors()->first());
        }
        try {

            DB::beginTransaction();
            $user = $this->userRepository->store($request);
            $user->update([
                'code' => 1111,
                // 'code' => rand(1000, 9999),
            ]);
            // $sendSMS = $this->forJawalyService->sendSMS($user->mobile, $user->code);
            // if ($sendSMS['code'] === 200) {
            if (true) {
                DB::commit();               
            } else {
                DB::rollback();
                return responseJson(500, 'there is some thing wrong , please contact technical support');
            }
        } catch (\Exception $e) {
            DB::rollback();
            return responseJson(500, 'there is some thing wrong , please contact technical support');
        }
        return responseJson(200, "success");
    }

    public function mobileCheck(Request $request)
    {

        $validator = Validator::make($request->all(), [
            'code' => 'required|exists:users,code|max:4',
            'mobile' => 'required|exists:users,mobile|max:20|min:6',
        ]);
        if ($validator->fails()) {
            return responseJson(400, "Bad Request", $validator->errors()->first());
        }
        try {
            $user = $this->user->where('mobile', $request->mobile)->where('code', $request->code)->first();
            if(!is_null($user->deleted_at)){
                return responseJson(401, "This Account Not Activate , Please Contact Technical Support");
            }
            DB::beginTransaction();
            $user->update([
                'code' => null,
                'mobile_verified_at' =>  now(),
            ]);
            $user->token = JWTAuth::customClaims(['exp' => Carbon::now()->addYears(20)->timestamp])->fromUser($user);
            DB::commit();
        } catch (\Exception $e) {
            DB::rollback();
            return responseJson(500, "Internal Server Error");
        }
        // except
        return responseJson(200, "success", $user->makeHidden(['password', 'code'])->load(['languages']));
    }

    public function regenerateCode(Request $request)
    {

        $validator = Validator::make($request->all(), [
            'mobile' => 'required|exists:users,mobile|max:20|min:6',
        ]);
        if ($validator->fails()) {
            return responseJson(400, "Bad Request", $validator->errors()->first());
        }

        $user = $this->user->where('mobile', $request->mobile)->first();
        if(!is_null($user->deleted_at)){
            return responseJson(401, "This Account Not Activate , Please Contact Technical Support");
        }
        try {

            DB::beginTransaction();
            $user->update([
                'code' => 1111,
                // 'code' => rand(1000, 9999),
                'mobile_verified_at' =>  null,
            ]);
            // DB::commit();
            // $sendSMS = $this->forJawalyService->sendSMS($user->mobile, $user->code);
            // if ($sendSMS['code'] === 200) {
            if (true) {
                DB::commit();
            } else {
                DB::rollback();
                return responseJson(500, 'there is some thing wrong , please contact technical support');
            }
        } catch (\Exception $e) {
            DB::rollback();
            return responseJson(500, "Internal Server Error");
        }
        return responseJson(200, "success");
    }

    public function login(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'mobile' => 'required|exists:users,mobile|max:20|min:6',
                'password' => 'required',

                'fcm_token' => 'nullable|string',
            ]);
            if ($validator->fails()) {
                return responseJson(400, "Bad Request", $validator->errors()->first());
            }

            $user = $this->user->where('mobile', $request->mobile)->first();
            if(!$user || !is_null($user->deleted_at) || (int)$user->is_activate == 0){
                return responseJson(401, "هذا الحساب غير مفعل برجاء الاتصال بالاداره");
            }
            if(is_null($user->mobile_verified_at)){
                return responseJson(401, "هذا الحساب غير مؤكد برجاء تاكيد رقم الجوال");
            }
            if(!FacadesHash::check($request->password, $user->password)){
                return response()->json(['error' => 'خطا في رقم الجوال او الرقم السري'], 401);
            }

            if (isset($request->fcm_token) && !is_null($request->fcm_token)) {
                $user->update([
                    'fcm_token' => $request->fcm_token ?? null,
                ]);
            }
            $user->token = JWTAuth::customClaims(['exp' => Carbon::now()->addYears(20)->timestamp])->fromUser($user);
            return responseJson(200, "success", $user->makeHidden(['password', 'code'])->load(['languages']));
        } catch (\Exception $e) {
            return responseJson(500, "Internal Server Error {$e}");
        }
    }
    
    public function me()
    {
        return responseJson(200, "success", auth()->user()->makeHidden(['password', 'code'])->load(['languages', 'wallet.transactions', 'days', 'album', 'ratings.user']));
    }
    
    public function favorates()
    {
        $user = auth()->user();
        $data['favorate_guides'] = $user->favorate_guides;
        $data['favorate_services'] = $user->favorate_services()->with('serviceable')->get();
        return responseJson(200, "success", $data);
    }
    
    public function points()
    {
        $user = auth()->user()->load('points', 'unused_points');
        $data['points'] = $user?->points;
        $data['unused_points'] = $user?->unused_points;
        $data['sum_unused_points'] = array_sum(array_column($user?->unused_points->toArray(), 'points'));
        return responseJson(200, "success", $data);
    }
    
    public function notifications()
    {
        $user = auth()->user();
        if ((int)$user->user_type == 1) {
            $data['notifications'] = $user->client_notifications()->with('serviceable')->latest()->get();
        }else {
            $data['notifications'] = $user->provider_notifications()->with('serviceable')->latest()->get();
        }
        return responseJson(200, "success", $data);
    }
    
    public function removeFromFavorates(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'service_id' => 'required|integer', 
            'service_type' => 'required|string|in:Car,Destination,Restaurant,Hotel,Cafe,User', 
        ]);
        if ($validator->fails()) {
            return responseJson(400, "Bad Request", $validator->errors()->first());
        }
        $user = auth()->user();
        if (in_array($request->service_type, ["Car", "Destination", "Restaurant", "Hotel", "Cafe"])) {
            $user->favorate_services()->where('serviceable_type', 'App\Models\\' . $request->service_type)->where('serviceable_id', $request->service_id)->delete();
        } else if (in_array($request->service_type, ["User"])) {
            $user->favorate_guides()->where('guide_id', $request->service_id)->detach();
        }
        return responseJson(200, "success");
    }

    public function userUpdate(Request $request)
    {

        $user = auth()->user();
        $reqIf = (int)$user->user_type == 1 ? 'nullable' : 'required'; 
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:254|unique:users,name,' . $user->id,
            'email' => 'required|string|max:254|unique:users,email,' . $user->id,
            
            'file' => 'nullable|image|mimes:jpeg,png,jpg,gif',
            'day_price' => $reqIf . '|string|max:254', 
            'hour_price' => $reqIf . '|string|max:254', 
            'info' => $reqIf . '|string|max:25000', 
            'experience' => $reqIf . '|string|max:25000', 
            
            'country_id' => 'nullable|exists:countries,id',
            'ibna' => 'nullable|max:255',
            'logo' => 'nullable|image|mimes:jpeg,png,jpg,gif',
            'documented_files' => 'nullable|array',
            'documented_files.*' => 'required|image|mimes:jpeg,png,jpg,gif,webp',

            'interests' => $reqIf  . '|array',
            'interests.*' => $reqIf  . '|string',
            'tours_type' => $reqIf  . '|array',
            'tours_type.*' => $reqIf  . '|string',
            'languages' => $reqIf  . '|array',
            'languages.*' => $reqIf  . '|exists:languages,id',

            'days' => $reqIf . '|array',
            'days.*.day' => $reqIf . '|in:السبت,الاحد,الاثنين,الثلاثاء,الاربعاء,الخميس,الجمعه',
            'days.*.start' => 'nullable|date_format:H:i',
            'days.*.end' => 'nullable|date_format:H:i',
            'days.*.day_status' => 'nullable|in:0,1,2',
        ]);
        if ($validator->fails()) {
            return responseJson(400, "Bad Request", $validator->errors()->first());
        }
        $user = $this->userRepository->customUpdate($request, $user->id);
        return responseJson(200, "success", $user->makeHidden(['password', 'code'])->load(['languages', 'days']));
    }

    public function guideAlbum(Request $request)
    {
        $user = auth()->user();
        $validator = Validator::make($request->all(), [
            'album' => 'required|array',
            'album.*' => 'required|mimes:jpeg,png,jpg,gif,svg,webp,mp4,mov,avi,wmv,mkv|max:100240',
        ]);
        if ($validator->fails()) {
            return responseJson(400, "Bad Request", $validator->errors()->first());
        }
        $user = $this->userRepository->guideAlbum($request, $user->id);
        return responseJson(200, "success", $user->makeHidden(['password', 'code'])->load(['languages', 'days', 'album']));
    }

    public function changePassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'old_password' => 'required',  
            'new_password' => 'required|max:30|confirmed',  
        ]);

        if ($validator->fails()) {
            return responseJson(400, "Bad Request", $validator->errors()->first());
        }

        $user = auth()->guard('api')->user();

        if (!$user) {
            return responseJson(401, "Unauthorized: User not logged in.");
        }

        if (!Hash::check($request->old_password, $user->password)) {
            return responseJson(400, "Old password is incorrect.");
        }

        try {
            $user->update([
                'password' => bcrypt($request->new_password),
            ]);
        } catch (\Exception $e) {
            return responseJson(500, "Internal Server Error", $e->getMessage());
        }

        return responseJson(200, "Password changed successfully.");
    }

    // this function => "mobileCheck" after this step 
    public function changeMobileNum(Request $request)
    {
        
        $validator = Validator::make($request->all(), [
            'mobile' => 'required|unique:users,mobile|max:20|min:6'
        ]);
        if ($validator->fails()) {
            return responseJson(400, "Bad Request", $validator->errors()->first());
        }
  
        $user = auth()->user(); 
        try {

            DB::beginTransaction();
            $user->update([
                'mobile' => $request->mobile,
                'mobile_verified_at' =>  null,
                'code' => 1111,
                // 'code' => rand(1000, 9999),
            ]);            
            // DB::commit();
            // $sendSMS = $this->forJawalyService->sendSMS($user->mobile, $user->code);
            // if ($sendSMS['code'] === 200) {
            if (true) {
                DB::commit();
            } else {
                DB::rollback();
                return responseJson(500, 'there is some thing wrong , please contact technical support');
            }
        } catch (\Exception $e) {
            DB::rollback();
            return responseJson(500, "Internal Server Error");
        }
        return responseJson(200, "success");
    }

    public function logout()
    {
        auth()->logout();
        $data['token'] = null;
        return responseJson(200, "successfully logged out", $data);
    }

    public function refresh()
    {
        return responseJson(200, "success", auth()->refresh());
    }
    
    public function sendResetCodePassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'mobile' => 'required|max:20|min:6',
        ]);
        if ($validator->fails()) {
            return responseJson(400, "Bad Request", $validator->errors()->first());
        }

        $user = $this->user->where('mobile', $request->mobile)->first();
        if (!$user) {
            return responseJson(404, "User not found");
        }

        try {
            $user->update(['code' => 1111]);

        } catch (\Exception $e) {
            return responseJson(500, "Internal Server Error");
        }

        return responseJson(200, "Reset code sent successfully.");
    }

    public function verifyResetCodePassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'mobile' => 'required|exists:users,mobile|max:20|min:6',
            'code' => 'required|exists:users,code|max:4',
        ]);
        if ($validator->fails()) {
            return responseJson(400, "Bad Request", $validator->errors()->first());
        }

        $user = $this->user->where('mobile', $request->mobile)->first();
        if (!$user || !is_null($user->deleted_at) || $user->code != $request->code) {
            return responseJson(401, "There Is Something Wrong, Please Contact Technical Support");
        }

        return responseJson(200, "Code verified successfully.");
    }

    public function resetPassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'mobile' => 'required|exists:users,mobile|max:20|min:6',
            'code' => 'required|exists:users,code|max:4',
            'password' => 'required|confirmed|max:30',
        ]);
        if ($validator->fails()) {
            return responseJson(400, "Bad Request", $validator->errors()->first());
        }

        $user = $this->user->where('mobile', $request->mobile)->first();
        if (!$user || !is_null($user->deleted_at) || $user->code != $request->code) {
            return responseJson(401, "This Account Not Activated, Please Contact Technical Support");
        }

        try {
            $user->update([
                'password' => bcrypt($request->password),
                'code' => null, 
            ]);
        } catch (\Exception $e) {
            return responseJson(500, "Internal Server Error");
        }
        return responseJson(200, "Password reset successfully.");
    }

    public function deleteAccount()
    {
        $user = auth()->guard('api')->user();
        $user->deleted_at = date("Y-m-d h:m:s");
        $user->save();
        return responseJson(200, "success");
    }
    
    public function statistics(Request $request)
    {

        $auth = auth()->user();
        $period = request('period', '6m'); // '1m', '6m', '1y'
        // dd($period);
        switch ($period) {
            case '1m':
                $from = Carbon::now()->subMonth();
                break;
            case '1y':
                $from = Carbon::now()->subYear();
                break;
            default:
                $from = Carbon::now()->subMonths(6);
                break;
        }
        // dd($from);
        // dd($auth);
        $number_of_tourists_served = 0;
        switch ((int)$auth->user_type) {
            case 2:
                $for_rating = User::class;
                $orders_type = 'guide_orders';
                $profitsQuery = $this->guideOrder->where('guide_id', $auth->id);
                $number_of_tourists_served = $auth->guide_orders()->whereIn('status', [3])->with('chat_group.members')->get()->pluck('chat_group.members')->flatten()->unique('user_id')->count();
                break;
            
            case 3:
                $for_rating = Hotel::class;
                $orders_type = 'hotel_orders';
                $profitsQuery = $this->hotelOrder->where('provider_id', $auth->id);
                break;

            case 1:
                $for_rating = Car::class;
                $orders_type = 'car_orders';
                $profitsQuery = $this->carOrder->where('provider_id', $auth->id);
                break;
                
            case 5:
                $for_rating = Cafe::class;
                $orders_type = 'cafe_orders';
                $profitsQuery = $this->cafeOrder->where('provider_id', $auth->id);
                break;
            
            case 6:
                $for_rating = Restaurant::class;
                $orders_type = 'restaurant_orders';
                $profitsQuery = $this->restaurantOrder->where('provider_id', $auth->id);
                break;
            
            default:
                return responseJson(500, 'there is some thing wrong , please contact technical support');
                break;
        }

        $usr_statistics = $auth->loadCount([
            "$orders_type as new_orders_count" => fn($q) => $q->whereIn('status', [1, 2]),
            "$orders_type as completed_orders_count" => fn($q) => $q->where('status', 3),
        ])->loadSum([
            "$orders_type as profits_sum" => fn ($q) => $q->where('status', 3),
        ], 'cost');
        
        $ratings = $this->rating->where('is_activate', 1)
        ->whereHasMorph('serviceable', [$for_rating], function ($q) use ($auth) {
            if (!in_array((int) $auth->user_type, [1, 2])) {
                $q->where('owner_id', $auth->id);
            }
        });

        $profits_months = $profitsQuery->where('status', 3)->where('created_at', '>=', $from)
        ->selectRaw('YEAR(created_at) as year, MONTH(created_at) as month, SUM(cost) as total')
        ->groupBy('year', 'month')->orderBy('year')->orderBy('month')
        ->get();

        $statistics['count_rating'] = (clone $ratings)->count(); // عدد المراجعات
        $statistics['avg_rating'] = round((clone $ratings)->avg('rating') ?: 0, 1); // متوسط التقييم (مثلاً 4.8)
        $statistics['new_orders_count'] = $usr_statistics['new_orders_count'];
        $statistics['completed_orders_count'] = $usr_statistics['completed_orders_count'];
        $statistics['number_of_tourists_served'] = $number_of_tourists_served;
        $statistics['profits'] = (int)$usr_statistics['profits_sum'];
        $statistics['profits_months'] = $profits_months;
        return $statistics;
    }

}